import React, { useEffect, useState } from "react";
import * as api from "../lib/api";
import { useAuth } from "../context/AuthContext";
import { Link } from "react-router-dom";
import { Shield, ArrowLeft, Check, X } from "lucide-react";

export default function AdminPanel() {
    const [users, setUsers] = useState({});
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);
    const { logout } = useAuth();

    useEffect(() => {
        loadUsers();
    }, []);

    const loadUsers = async () => {
        try {
            setLoading(true);
            const data = await api.getAdminUsers();
            setUsers(data);
        } catch (err) {
            setError(err.message);
        } finally {
            setLoading(false);
        }
    };

    const handleApproval = async (uid, approved) => {
        try {
            await api.adminApproveUser(uid, approved);
            // Optimistic update
            setUsers(prev => ({
                ...prev,
                [uid]: { ...prev[uid], approved }
            }));
        } catch (err) {
            alert("Failed to update user: " + err.message);
        }
    };

    if (loading) return <div className="p-8">Loading users...</div>;
    if (error) return <div className="p-8 text-red-500">Error: {error}</div>;

    return (
        <div className="app p-8" style={{ maxWidth: "1000px", margin: "0 auto" }}>
            <div className="flex justify-between items-center mb-8">
                <div className="flex items-center gap-4">
                    <Link to="/" className="btn-secondary"><ArrowLeft size={16} /> Back to Dashboard</Link>
                    <h1 className="text-2xl font-bold flex items-center gap-2">
                        <Shield className="text-accent-primary" /> User Management
                    </h1>
                </div>
                <button onClick={logout} className="btn-secondary">Sign Out</button>
            </div>

            <div className="bg-[var(--bg-secondary)] rounded-lg overflow-hidden">
                <table className="w-full text-left" style={{ borderCollapse: "collapse" }}>
                    <thead>
                        <tr className="border-b border-[var(--border-color)]">
                            <th className="p-4">Email</th>
                            <th className="p-4">Role</th>
                            <th className="p-4">Status</th>
                            <th className="p-4">Joined</th>
                            <th className="p-4 text-right">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        {Object.entries(users).map(([uid, user]) => (
                            <tr key={uid} className="border-b border-[var(--border-color)] last:border-0 hover:bg-[var(--bg-tertiary)]">
                                <td className="p-4">{user.email}</td>
                                <td className="p-4">
                                    <span className={`px-2 py-1 rounded text-xs ${user.role === 'admin' ? 'bg-purple-900 text-purple-200' : 'bg-gray-700'}`}>
                                        {user.role}
                                    </span>
                                </td>
                                <td className="p-4">
                                    <span className={`px-2 py-1 rounded text-xs ${user.approved ? 'bg-green-900 text-green-200' : 'bg-yellow-900 text-yellow-200'}`}>
                                        {user.approved ? 'Approved' : 'Pending'}
                                    </span>
                                </td>
                                <td className="p-4 text-sm text-[var(--text-muted)]">
                                    {new Date(user.createdAt).toLocaleDateString()}
                                </td>
                                <td className="p-4 text-right">
                                    {user.role !== 'admin' && (
                                        <div className="flex gap-2 justify-end">
                                            {!user.approved ? (
                                                <button
                                                    onClick={() => handleApproval(uid, true)}
                                                    className="btn-success text-xs flex items-center gap-1"
                                                >
                                                    <Check size={12} /> Approve
                                                </button>
                                            ) : (
                                                <button
                                                    onClick={() => handleApproval(uid, false)}
                                                    className="btn-danger text-xs flex items-center gap-1"
                                                >
                                                    <X size={12} /> Revoke
                                                </button>
                                            )}
                                        </div>
                                    )}
                                </td>
                            </tr>
                        ))}
                    </tbody>
                </table>
            </div>
        </div>
    );
}
